<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use App\Enums\Gender;
use Filament\Models\Contracts\FilamentUser;
use Filament\Panel;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable implements FilamentUser
{
    use HasApiTokens, HasFactory, Notifiable;
    use softDeletes;

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $guarded = [];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'national_id' => 'integer',
        'gender' => Gender::class,
        'is_trainee' => 'boolean',
        'is_trainer' => 'boolean',
        'max_courses_per_year' => 'integer',
    ];

    public function canAccessPanel(Panel $panel): bool
    {
        return true;
    }

    public function office(): BelongsTo
    {
        return $this->belongsTo(Office::class)->withTrashed();
    }

    public function school(): BelongsTo
    {
        return $this->belongsTo(School::class)->withTrashed();
    }

    public function trainerCourses(): BelongsToMany
    {
        return $this->belongsToMany(Course::class,
            (new CourseTrainer())->getTable(),
            'trainer_id',
            'course_id',
        )
            ->withPivot('id')
            ->withTimestamps();
    }

    public function traineeCourses(): BelongsToMany
    {
        return $this->belongsToMany(
            Course::class,
            (new CourseTrainee())->getTable(),
            'trainee_id',
            'course_id'
        )
            ->withPivot('id')
            ->withTimestamps();
    }

    public function eventCertificate(): BelongsToMany
    {
        return $this->belongsToMany(CertificateEvent::class, 'certificate_event_users')->withPivot('id');
    }

    public function scopeTrainer($query)
    {
        return $query->where('is_trainer', true);
    }

    public function scopeTrainee($query)
    {
        return $query->where('is_trainee', true);
    }

    public function scopeMale($query)
    {
        return $query->where('gender', Gender::MALE);
    }

    public function scopeFemale($query)
    {
        return $query->where('gender', Gender::FEMALE);
    }

    public function scopeEduJob($query)
    {
        return $query->where('has_edu_job', true);
    }

    public function scopeManagerJob($query)
    {
        return $query->where('has_manager_job', true);
    }

    public function scopeAdminControl(Builder $builder): Builder
    {
        $admin = auth('admin')->user();

        // Handles the OR condition between Male and Female
        $builder
            ->where(function ($query) use ($admin) {
                if ($admin->is_has_males) {
                    $query->orWhere('gender', Gender::MALE);
                }
                if ($admin->is_has_females) {
                    $query->orWhere('gender', Gender::FEMALE);
                }
            });

        // Handles the AND condition with Jobs
        return $builder
            ->when($admin->is_has_edu_job, function ($query) {
                return $query->where('has_edu_job', true);
            })
            ->when($admin->is_has_manager_job, function ($query) {
                return $query->orWhere('has_manager_job', true);
            });
    }

    public function getAttendancePercentageForCourse(Course $course): float
    {
        $courseDays = $course->getDays()->count();
        $attendancesCount = $course->attendances()
            ->where('is_attended', true)
            ->where('trainee_id', $this->id)
            ->count();

        return $courseDays > 0 ? ($attendancesCount / $courseDays) * 100 : 0;
    }

    public function canGetHisCertifcateForCourse(Course $course): bool
    {
        return $this->getAttendancePercentageForCourse($course) >= 70;
    }
}
