<?php

namespace App\Filament\SuperAdmin\Resources\CourseResource\RelationManagers;

use App\Exports\CourseTrainersExport;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use pxlrbt\FilamentExcel\Actions\Tables\ExportAction;
use pxlrbt\FilamentExcel\Actions\Tables\ExportBulkAction;

class TrainersRelationManager extends RelationManager
{
    protected static string $relationship = 'trainers';

    protected static ?string $inverseRelationship = 'trainerCourses';

    protected static ?string $recordTitleAttribute = 'name';

    public static function getTitle(\Illuminate\Database\Eloquent\Model $ownerRecord, string $pageClass): string
    {
        return __('admin.relation.trainers');
    }

    /**
     * @deprecated Use `getModelLabel()` instead.
     */
    public static function getPluralRecordLabel(): ?string
    {
        return __('admin.relation.trainers');
    }

    /**
     * @deprecated Use `getModelLabel()` instead.
     */
    public static function getRecordLabel(): ?string
    {
        return __('admin.relation.trainer');
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('national_id')
                    ->required()
                    ->numeric()
                    ->autofocus()
                    ->unique(ignoreRecord: true)
                    ->label(__('trainer.national_id'))
                    ->placeholder(__('trainer.national_id_placeholder'))
                    ->helperText(__('trainer.national_id_helper_text')),

                TextInput::make('name')
                    ->required()
                    ->autofocus()
                    ->label(__('trainer.name'))
                    ->placeholder(__('trainer.name_placeholder'))
                    ->helperText(__('trainer.name_helper_text')),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('name')
            ->columns([
                TextColumn::make('id')
                    ->sortable()
                    ->searchable()
                    ->label('ID'),
                TextColumn::make('national_id')
                    ->sortable()
                    ->searchable()
                    ->label(__('trainer.national_id')),

                TextColumn::make('name')
                    ->sortable()
                    ->searchable()
                    ->label(__('trainer.name')),

                TextColumn::make('email')
                    ->sortable()
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->label(__('trainer.email')),

                TextColumn::make('phone')
                    ->sortable()
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->label(__('trainer.phone')),

                Tables\Columns\TextColumn::make('office.name')
                    ->sortable()
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->label(__('trainer.office')),

                Tables\Columns\TextColumn::make('school.name')
                    ->sortable()
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->label(__('trainer.school')),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->headerActions([
                // Tables\Actions\CreateAction::make(),
                Tables\Actions\AttachAction::make()
                    ->recordSelectSearchColumns(['national_id', 'name', 'email', 'phone'])
                    ->recordSelectOptionsQuery(function (Builder $query) {
                        $query->where('is_trainer', true);
                    }),

                ExportAction::make()->exports([
                    CourseTrainersExport::make()
                        ->askForFilename(
                            default: $this->ownerRecord->title.' - trainers - '.date('Y-m-d').' - export',
                            label : __('action.export_file_name')
                        ),
                ]),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                //  Tables\Actions\EditAction::make(),
                Tables\Actions\DetachAction::make(),
                //                Tables\Actions\DeleteAction::make(),
                //                Tables\Actions\ForceDeleteAction::make(),
                //                Tables\Actions\RestoreAction::make(),

                Tables\Actions\Action::make('view')
                    ->label(__('admin.view_certificate'))
                    ->button()
                    ->url(function ($record) {
                        return route('admin.trainer.certificate', ['course' => $this->ownerRecord, 'user' => $record]);
                    }, shouldOpenInNewTab: true),

                Tables\Actions\Action::make('download')
                    ->label(__('admin.download_certificate'))
                    ->button()
                    ->url(function ($record) {
                        return route('admin.trainer.certificate', ['course' => $this->ownerRecord, 'user' => $record, 'download' => true]);
                    }, shouldOpenInNewTab: true),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DetachBulkAction::make(),
                    //                    Tables\Actions\DeleteBulkAction::make(),
                    //                    Tables\Actions\RestoreBulkAction::make(),
                    //                    Tables\Actions\ForceDeleteBulkAction::make(),

                    ExportBulkAction::make()->exports([
                        CourseTrainersExport::make()
                            ->askForFilename(
                                default: $this->ownerRecord->title.' - trainers - '.date('Y-m-d').' - export',
                                label : __('action.export_file_name')
                            ),
                    ]),
                ]),
            ])
            ->emptyStateActions([
                // Tables\Actions\CreateAction::make(),
                Tables\Actions\AttachAction::make(),
            ])
            ->modifyQueryUsing(fn (Builder $query) => $query->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]));
    }
}
