<?php

namespace App\Filament\SuperAdmin\Resources\CertificateEventResource\RelationManagers;

use App\Exports\UsersExport;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use pxlrbt\FilamentExcel\Actions\Tables\ExportAction;

class UsersRelationManager extends RelationManager
{
    protected static string $relationship = 'users';

    protected static ?string $inverseRelationship = 'eventCertificate';

    public static function getTitle(\Illuminate\Database\Eloquent\Model $ownerRecord, string $pageClass): string
    {
        return __('admin.relation.users');
    }

    /**
     * @deprecated Use `getModelLabel()` instead.
     */
    public static function getPluralRecordLabel(): ?string
    {
        return __('admin.relation.users');
    }

    /**
     * @deprecated Use `getModelLabel()` instead.
     */
    public static function getRecordLabel(): ?string
    {
        return __('admin.relation.user');
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('national_id')
                    ->required()
                    ->numeric()
                    ->autofocus()
                    ->unique(ignoreRecord: true)
                    ->label(__('trainer.national_id'))
                    ->placeholder(__('trainer.national_id_placeholder'))
                    ->helperText(__('trainer.national_id_helper_text')),

                TextInput::make('name')
                    ->required()
                    ->autofocus()
                    ->label(__('trainer.name'))
                    ->placeholder(__('trainer.name_placeholder'))
                    ->helperText(__('trainer.name_helper_text')),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('name')
            ->columns([
                TextColumn::make('id')
                    ->sortable()
                    ->searchable()
                    ->label('ID'),

                TextColumn::make('national_id')
                    ->sortable()
                    ->searchable()
                    ->label(__('input.national_id')),

                TextColumn::make('name')
                    ->sortable()
                    ->searchable()
                    ->label(__('input.name')),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
            ])
            ->headerActions([
                Tables\Actions\AttachAction::make()
                    ->recordSelectSearchColumns(['national_id', 'name', 'email', 'phone']),

                ExportAction::make()->exports([
                    UsersExport::make()
                        ->askForFilename(
                            default: $this->ownerRecord->title.' - event users - '.date('Y-m-d').' - export',
                            label : __('action.export_file_name')
                        ),
                ]),
            ])
            ->actions([
                Tables\Actions\DetachAction::make(),

                Tables\Actions\Action::make('view')
                    ->label(__('admin.view_certificate'))
                    ->button()
                    ->url(function ($record) {
                        return route('admin.event.certificate', ['event' => $this->ownerRecord, 'user' => $record]);
                    }, shouldOpenInNewTab: true),

                Tables\Actions\Action::make('download')
                    ->label(__('admin.download_certificate'))
                    ->button()
                    ->url(function ($record) {
                        return route('admin.event.certificate', ['event' => $this->ownerRecord, 'user' => $record, 'download' => true]);
                    }, shouldOpenInNewTab: true),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DetachBulkAction::make(),
                ]),
            ])
            ->modifyQueryUsing(fn (Builder $query) => $query->withoutGlobalScopes([
                SoftDeletingScope::class,
            ]));
    }
}
