<?php

namespace App\Filament\Employee\Resources\Trainee\CourseResource\Pages;

use App\Filament\Employee\Resources\Trainee\CourseResource;
use App\Filament\Employee\Resources\Trainee\CourseResource\Widgets\StatsOverview;
use Filament\Actions\Action;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\ViewRecord;
use Filament\Support\Enums\ActionSize;

class ViewCourse extends ViewRecord
{
    protected static string $resource = CourseResource::class;

    protected function getHeaderWidgets(): array
    {
        return [
            StatsOverview::make(),
        ];
    }

    protected function getHeaderActions(): array
    {
        return [
            $this->joinAction(),
            $this->leaveAction(),
        ];
    }

    private function joinAction(): Action
    {
        return Action::make('join_course')
            ->label(__('course.join_btn'))
            ->size(ActionSize::Large)
            ->icon('heroicon-m-plus')
            ->visible(! auth()->user()->traineeCourses()->where('course_id', $this->record->id)->exists())
            ->disabled(! $this->record->stillCanJoinOrLeave())
            ->tooltip(function () {
                $last_time_to_join = $this->record->last_time_to_join;

                if ($last_time_to_join->isPast()) {
                    return __('course.ended_join_btn_tooltip');
                }

                return __('course.join_btn_tooltip', ['date' => $last_time_to_join->format('d/m/Y H:i')]);
            })
            ->outlined()
            ->color('primary')
            ->action(function () {
                $course = $this->record;
                $last_time_to_join = $course->last_time_to_join;

                if ($last_time_to_join->isPast()) {
                    Notification::make()
                        ->title(__('course.ended_join_btn_tooltip'))
                        ->danger()
                        ->body(__('course.ended_join_btn_tooltip'))
                        ->send();

                    return;
                }

                $user = auth()->user();

                if ($course->trainees()->count() >= $course->max_trainees) {
                    Notification::make()
                        ->title(__('course.max_trainees_reached'))
                        ->danger()
                        ->body(__('course.max_trainees_reached'))
                        ->send();

                    return;
                }

                $userSchool = $user->school;

                if (! $userSchool) {
                    Notification::make()
                        ->title(__('course.no_school'))
                        ->danger()
                        ->body(__('course.no_school'))
                        ->send();

                    return;
                }

                $max_number = $course->max_trainees_per_school;

                if ($course->trainees()->where('school_id', $userSchool->id)->count() >= $max_number) {
                    Notification::make()
                        ->title(__('course.max_trainees_reached'))
                        ->danger()
                        ->body(__('course.max_trainees_reached_of_school'))
                        ->send();

                    return;
                }

                // every trainee can join only [number] course per year
                $year = now()->year;
                $max_courses_per_year = $user->max_courses_per_year;
                $traineeCourses = $user->traineeCourses()->whereYear('course_trainees.created_at', '=', $year)->count();

                if ($traineeCourses >= $max_courses_per_year) {
                    Notification::make()
                        ->title(__('course.max_courses_per_year_reached'))
                        ->danger()
                        ->body(__('course.max_courses_per_year_reached'))
                        ->send();

                    return;
                }

                // if trainee already joined this course
                if ($user->traineeCourses()->where('course_id', $course->id)->exists()) {
                    Notification::make()
                        ->title(__('course.already_joined'))
                        ->danger()
                        ->body(__('course.already_joined'))
                        ->send();

                    return;
                }

                // if course is online then every trainee can join
                if ($course->isOnline()) {
                    $user->traineeCourses()->attach($course->id);

                    Notification::make()
                        ->title(__('course.joined'))
                        ->success()
                        ->body(__('course.joined'))
                        ->send();

                    return;
                }

                // if course is for_all_offices then all can join , else only trainees from same office can join

                if ($course->is_for_all_offices) {
                    $user->traineeCourses()->attach($course->id);

                    Notification::make()
                        ->title(__('course.joined'))
                        ->success()
                        ->body(__('course.joined'))
                        ->send();

                    return;
                }

                $userOffice = $user->office;

                if (! $userOffice) {
                    Notification::make()
                        ->title(__('course.no_office'))
                        ->danger()
                        ->body(__('course.no_office'))
                        ->send();

                    return;
                }

                if ($course->offices()->where('offices.id', $userOffice->id)->exists()) {
                    $user->traineeCourses()->attach($course->id);

                    Notification::make()
                        ->title(__('course.joined'))
                        ->success()
                        ->body(__('course.joined'))
                        ->send();

                    return;
                }
            })
            ->requiresConfirmation();
    }

    private function leaveAction(): Action
    {
        return Action::make('leave_course')
            ->label(__('course.leave_btn'))
            ->size(ActionSize::Large)
            ->icon('heroicon-m-minus')
            ->visible(auth()->user()->traineeCourses()->where('course_id', $this->record->id)->exists())
            ->disabled(! $this->record->stillCanJoinOrLeave())
            ->tooltip(function () {
                $last_time_to_leave = $this->record->last_time_to_join;

                if ($last_time_to_leave->isPast()) {
                    return __('course.ended_lave_btn_tooltip');
                }

                return __('course.leave_btn_tooltip', ['date' => $last_time_to_leave->format('d/m/Y H:i')]);
            })
            ->outlined()
            ->color('danger')
            ->action(function () {
                $course = $this->record;
                $last_time_to_leave = $course->last_time_to_join;

                if ($last_time_to_leave->isPast()) {
                    Notification::make()
                        ->title(__('course.ended_lave_btn_tooltip'))
                        ->danger()
                        ->body(__('course.ended_lave_btn_tooltip'))
                        ->send();

                    return;
                }

                $user = auth()->user();

                if (! $user->traineeCourses()->where('course_id', $course->id)->exists()) {
                    Notification::make()
                        ->title(__('course.not_joined'))
                        ->danger()
                        ->body(__('course.not_joined'))
                        ->send();

                    return;
                }

                $user->traineeCourses()->detach($course->id);

                Notification::make()
                    ->title(__('course.left'))
                    ->success()
                    ->body(__('course.left'))
                    ->send();
            })
            ->requiresConfirmation();
    }
}
