<?php

namespace App\Filament\Admin\Resources\CourseResource\Pages;

use App\Filament\Admin\Resources\CourseResource;
use App\Models\Course;
use App\Models\CourseType;
use Filament\Actions;
use Filament\Resources\Pages\ListRecords;
use Konnco\FilamentImport\Actions\ImportAction;
use Konnco\FilamentImport\ImportField;

class ListCourses extends ListRecords
{
    protected static string $resource = CourseResource::class;

    public function getTabs(): array
    {
        $courseTypesThatHasMaxNumberOfCourses = CourseType::withCount('courses')->orderBy('courses_count', 'desc')->take(5)->get();

        $tabs = [
            null => ListRecords\Tab::make(__('filters.menu.all')),
        ];

        foreach ($courseTypesThatHasMaxNumberOfCourses as $courseType) {
            $tabs[$courseType->id] = ListRecords\Tab::make($courseType->name)->query(fn ($query) => $query->where('course_type_id', $courseType->id));
        }

        return $tabs;
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\CreateAction::make(),
            $this->addImportAction(),
        ];
    }

    private function addImportAction()
    {
        $admin = auth('admin')->user();

        return ImportAction::make('upload_courses_list')
            ->label(__('action.upload_courses_list'))
            ->fields([
                ImportField::make('course_type_id')
                    ->required()
                    ->rules(['exists:course_types,id'])
                    ->label(__('input.type')),

                ImportField::make('course_field_id')
                    ->required()
                    ->rules(['exists:course_fields,id'])
                    ->label(__('input.field')),

                ImportField::make('hours')
                    ->required()
                    ->rules(['numeric'])
                    ->label(__('input.hours')),

                ImportField::make('max_trainees')
                    ->required()
                    ->rules(['numeric'])
                    ->label(__('input.max_trainees')),

                ImportField::make('max_trainees_per_school')
                    ->required()
                    ->rules(['numeric'])
                    ->label(__('input.max_trainees_per_school')),

                // example 2023/09/11 00:02:00
                ImportField::make('start_at')
                    ->required()
                    ->rules(['date:Y/m/d H:i:s'])
                    ->label(__('input.start_at')),

                ImportField::make('end_at')
                    ->required()
                    ->rules(['date:Y/m/d H:i:s', 'after:start_at'])
                    ->label(__('input.end_at')),

                ImportField::make('title')
                    ->required()
                    ->rules(['string', 'max:255'])
                    ->label(__('input.title')),

                ImportField::make('is_online')
                    ->required()
                    ->rules(['boolean'])
                    ->label(__('input.is_online')),

                ImportField::make('is_for_boys')
                    ->required()
                    ->rules(['boolean'])
                    ->label(__('input.is_for_boys')),

                ImportField::make('is_for_girls')
                    ->required()
                    ->rules(['boolean'])
                    ->label(__('input.is_for_girls')),

                ImportField::make('is_for_edu_job')
                    ->required()
                    ->rules(['boolean'])
                    ->label(__('input.is_for_edu_job')),

                ImportField::make('is_for_manager_job')
                    ->required()
                    ->rules(['boolean'])
                    ->label(__('input.is_for_manager_job')),

                ImportField::make('is_for_all_offices')
                    ->required()
                    ->rules(['boolean'])
                    ->label(__('input.is_for_all_offices')),

                ImportField::make('meeting_link')
                    ->label(__('input.meeting_link')),

                ImportField::make('meeting_password')
                    ->label(__('input.meeting_password')),
            ])
            ->handleBlankRows(true)
            ->mutateBeforeCreate(function ($row) use ($admin) {
                $row = array_map('trim', $row);

                if (! $admin->is_has_males && $row['is_for_boys']) {
                    return null;
                }

                if (! $admin->is_has_females && $row['is_for_girls']) {
                    return null;
                }

                if (! $admin->is_has_edu_job && $row['is_for_edu_job']) {
                    return null;
                }

                if (! $admin->is_has_manager_job && $row['is_for_manager_job']) {
                    return null;
                }

                $row['start_at'] = date('Y-m-d H:i:s', strtotime($row['start_at']));
                $row['end_at'] = date('Y-m-d H:i:s', strtotime($row['end_at']));

                return $row;
            })
            ->handleRecordCreation(function ($row) {
                if (! $row) {
                    return null;
                }

                return Course::create($row);
            });
    }
}
